﻿// 折叠/展开功能实现
document.addEventListener('DOMContentLoaded', function() {
    // 获取当前页面的唯一标识（基于URL）
    const currentPageKey = 'llaPageState_' + window.location.pathname.replace(/\//g, '_');

    // 处理GRAM类的替换
    function replaceGramContent() {
        const gramElements = document.querySelectorAll('.GRAM');
        
        gramElements.forEach(gram => {
            // 保存原始HTML以备可能需要恢复
            if (!gram.dataset.original) {
                gram.dataset.original = gram.innerHTML;
            }
            
            // 获取文本内容
            let text = gram.textContent.trim();
            
            // 替换方括号
            text = text.replace(/\[/g, '⟨').replace(/\]/g, '⟩');
            
            // 替换语法术语，顺序不能乱
            const replacements = {
                'intransitive/transitive verb': 'I/T',
                'countable/uncountable noun': 'C/U',
                'intransitive phrasal verb': 'phrV I',
                'transitive phrasal verb': 'phrV T',
                'verb phrase': 'phrV',
                'uncountable noun': 'U',
                'countable noun': 'C',
                'intransitive verb': 'I',
                'transitive verb': 'T',
                'adjective': 'adj',
                'adverb': 'adv',
                'verb': 'v',
                'noun': 'n',
                'pronoun': 'pron',
                'preposition': 'prep',
                'conjunction': 'conj',
                'interjection': 'interj',
                'singular': 'sing',
                'plural': 'pl',
            };
            
            // 应用替换
            for (const [key, value] of Object.entries(replacements)) {
                if (text.includes(key)) {
                    text = text.replace(key, value);
                }
            }
            
            // 清理多余的空格
            text = text.replace(/\s+/g, ' ').trim();
            
            // 应用替换后的文本
            gram.textContent = text;
            
            // 添加样式
            gram.style.fontSize = '85%';
            gram.style.fontStyle = 'italic';
            gram.style.color = 'teal';
            gram.style.marginLeft = '6px';
            gram.style.display = 'inline-block';
        });
    }
    
    // 处理多单词词条的发音显示
    function handleMultiWordExp() {
        const expElements = document.querySelectorAll('.EXP');
        
        expElements.forEach(exp => {
            // 获取文本内容并计算单词数量
            const text = exp.textContent.trim();
            const wordCount = text.split(/\s+/).length;
            
            // 如果多于1个单词，标记并隐藏相邻的PRON
            if (wordCount > 1) {
                exp.setAttribute('data-multiword', 'true');
            } else {
                exp.setAttribute('data-multiword', 'false');
            }
        });
    }
    
    // 执行GRAM内容替换
    replaceGramContent();
    
    // 处理多单词词条的显示
    handleMultiWordExp();
    
    //第一组 处理References的折叠/展开
    const referenceTitles = document.querySelectorAll('.References .TITLE');
    
    referenceTitles.forEach(title => {
        // 添加手型光标
        title.style.cursor = 'pointer';
        
        // 找到对应的References容器
        const referencesContainer = title.closest('.References');
        
        // 收集所有需要折叠的元素（包括子元素和兄弟元素）
        const references = Array.from(referencesContainer.querySelectorAll('.Reference, .Crossref'));
        
        // 查找所有兄弟元素（在同一个父元素下的Reference和Crossref）
        let nextElement = referencesContainer.nextElementSibling;
        while (nextElement && (nextElement.classList.contains('Reference') || nextElement.classList.contains('Crossref'))) {
            references.push(nextElement);
            nextElement = nextElement.nextElementSibling;
        }
        
        // 移除TITLE元素本身
        const referencesToHide = references.filter(ref => ref !== title);
        
        // 默认折叠
        title.classList.add('collapsed');
        referencesToHide.forEach(ref => {
            ref.classList.add('collapsed');
        });
        
        // 添加点击事件
        title.addEventListener('click', function() {
            const isCollapsed = this.classList.contains('collapsed');
            
            if (isCollapsed) {
                // 展开
                this.classList.remove('collapsed');
                referencesToHide.forEach(ref => {
                    ref.classList.remove('collapsed');
                });
            } else {
                // 折叠
                this.classList.add('collapsed');
                referencesToHide.forEach(ref => {
                    ref.classList.add('collapsed');
                });
            }
            
            // 立即保存状态
            savePageState();
        });
    });    
    
    //第二组 处理SECDEF的折叠/展开 获取所有SECDEF元素
    const sectionHeaders = document.querySelectorAll('.SECDEF');
    // 为每个标题添加点击事件
    sectionHeaders.forEach(header => {
        // 添加手型光标
        header.style.cursor = 'pointer';

        // 添加单击事件（折叠/展开单个词条）
        header.addEventListener('click', function(e) {
            const isCollapsed = this.classList.contains('collapsed');
            
            if (isCollapsed) {
                // 展开操作
                this.classList.remove('collapsed');
                const currentSection = this.closest('.Section');
                const exponents = currentSection.querySelectorAll('.Exponent');
                exponents.forEach(exp => {
                    exp.classList.remove('collapsed');
                });
            } else {
                // 折叠操作
                this.classList.add('collapsed');
                const currentSection = this.closest('.Section');
                const exponents = currentSection.querySelectorAll('.Exponent');
                exponents.forEach(exp => {
                    exp.classList.add('collapsed');
                });
            }
            
            // 立即保存状态
            savePageState();
        });
        
        // 添加双击事件（折叠/展开全部词条）
        header.addEventListener('dblclick', function(e) {
            e.stopPropagation(); // 阻止事件冒泡
            
            // 记录双击时的滚动位置和锚点元素位置
            const scrollYBefore = window.scrollY;
            const anchorRect = this.getBoundingClientRect();
            const anchorTopRelativeToViewport = anchorRect.top;
            
            const isCollapsed = this.classList.contains('collapsed');
            const shouldExpandAll = isCollapsed; // 如果当前是折叠状态，双击后展开全部
            
            sectionHeaders.forEach(h => {
                if (shouldExpandAll) {
                    // 展开全部
                    h.classList.remove('collapsed');
                    const currentSection = h.closest('.Section');
                    const exponents = currentSection.querySelectorAll('.Exponent');
                    exponents.forEach(exp => {
                        exp.classList.remove('collapsed');
                    });
                } else {
                    // 折叠全部
                    h.classList.add('collapsed');
                    const currentSection = h.closest('.Section');
                    const exponents = currentSection.querySelectorAll('.Exponent');
                    exponents.forEach(exp => {
                        exp.classList.add('collapsed');
                    });
                }
            });
            
            // 在下一帧重新计算布局后，调整滚动位置以保持锚点位置不变
            requestAnimationFrame(() => {
                // 获取锚点元素的新位置
                const anchorRectAfter = this.getBoundingClientRect();
                const anchorTopRelativeToViewportAfter = anchorRectAfter.top;
                
                // 计算需要调整的滚动距离
                const scrollAdjustment = anchorTopRelativeToViewportAfter - anchorTopRelativeToViewport;
                
                // 调整滚动位置，保持双击位置不变
                window.scrollTo({
                    top: scrollYBefore + scrollAdjustment,
                    behavior: 'instant'
                });
                
                // 保存状态
                savePageState();
            });
        });
    });
    
    //第三组 处理EXP元素的折叠/展开
    const expElements = document.querySelectorAll('.EXP');
    
    expElements.forEach(exp => {
        // 添加手型光标
        exp.style.cursor = 'pointer';
        // 默认折叠状态将在applyDefaultCollapse中处理
        
        // 添加单击事件（折叠/展开单个EXP）
        exp.addEventListener('click', function(e) {
            e.stopPropagation(); // 阻止事件冒泡到父元素
            this.classList.toggle('collapsed');
            
            // 立即保存状态
            savePageState();
        });
        
        // 添加双击事件（折叠/展开当前SECDEF下的所有EXP）
        exp.addEventListener('dblclick', function(e) {
            e.stopPropagation(); // 阻止事件冒泡
            
            // 记录双击时的滚动位置和锚点元素位置
            const scrollYBefore = window.scrollY;
            const anchorRect = this.getBoundingClientRect();
            const anchorTopRelativeToViewport = anchorRect.top;
            
            // 找到当前EXP所在的SECDEF
            const currentSection = this.closest('.Section');
            if (!currentSection) return;
            
            // 找到当前SECDEF下的所有EXP元素
            const currentExps = currentSection.querySelectorAll('.EXP');
            
            // 检查点击的这个是什么状态
            const isCollapsed = this.classList.contains('collapsed');
            const shouldExpandAll = isCollapsed; // 如果当前是折叠状态，双击后展开全部
            
            // 切换所有EXP的状态
            currentExps.forEach(expElement => {
                if (isCollapsed) {
                    // 如果有折叠的，全部展开
                    expElement.classList.remove('collapsed');
                } else {
                    // 如果全部展开，全部折叠
                    expElement.classList.add('collapsed');
                }
            });
            
            // 在下一帧重新计算布局后，调整滚动位置以保持锚点位置不变
            requestAnimationFrame(() => {
                // 获取锚点元素的新位置
                const anchorRectAfter = this.getBoundingClientRect();
                const anchorTopRelativeToViewportAfter = anchorRectAfter.top;
                
                // 计算需要调整的滚动距离
                const scrollAdjustment = anchorTopRelativeToViewportAfter - anchorTopRelativeToViewport;
                
                // 调整滚动位置，保持双击位置不变
                window.scrollTo({
                    top: scrollYBefore + scrollAdjustment,
                    behavior: 'instant'
                });
                
                // 保存状态
                savePageState();
            });
        });
    });
    
    // 默认折叠函数（仅在无保存状态时执行）
    function applyDefaultCollapse() {
        // 第2组的默认全部折叠
        sectionHeaders.forEach(header => {
            header.classList.add('collapsed');
            const currentSection = header.closest('.Section');
            const exponents = currentSection.querySelectorAll('.Exponent');
            exponents.forEach(exp => {
                exp.classList.add('collapsed');
            });
        });
        
        // 第三组 EXP元素的默认折叠
        expElements.forEach(exp => {
            exp.classList.add('collapsed');
        });
    }
    
    // 临时展开所有元素的辅助函数
    function expandAllTemporarily(callback) {
        // 先展开所有元素
        document.querySelectorAll('.References .TITLE.collapsed').forEach(title => {
            title.classList.remove('collapsed');
            const referencesContainer = title.closest('.References');
            const references = Array.from(referencesContainer.querySelectorAll('.Reference, .Crossref'));
            let nextElement = referencesContainer.nextElementSibling;
            while (nextElement && (nextElement.classList.contains('Reference') || nextElement.classList.contains('Crossref'))) {
                references.push(nextElement);
                nextElement = nextElement.nextElementSibling;
            }
            const referencesToShow = references.filter(ref => ref !== title);
            referencesToShow.forEach(ref => {
                ref.classList.remove('collapsed');
            });
        });
        
        document.querySelectorAll('.SECDEF.collapsed').forEach(secdef => {
            secdef.classList.remove('collapsed');
            const currentSection = secdef.closest('.Section');
            const exponents = currentSection.querySelectorAll('.Exponent');
            exponents.forEach(exp => {
                exp.classList.remove('collapsed');
            });
        });
        
        document.querySelectorAll('.EXP.collapsed').forEach(exp => {
            exp.classList.remove('collapsed');
        });
        
        // 给一点时间让DOM更新，然后执行回调
        setTimeout(callback, 50);
    }
    
    // 页面状态保存与恢复功能
    // 状态存储对象
    const pageState = {
        scrollPosition: 0,
        collapsedElements: {
            references: [],
            secdefs: [],
            exponents: []
        }
    };

    // 保存页面状态到sessionStorage
    function savePageState() {
        // 保存滚动位置
        pageState.scrollPosition = window.scrollY || document.documentElement.scrollTop;
        
        // 保存References折叠状态
        pageState.collapsedElements.references = [];
        document.querySelectorAll('.References .TITLE').forEach(title => {
            if (title.classList.contains('collapsed')) {
                pageState.collapsedElements.references.push(title.textContent.trim());
            }
        });
        
        // 保存SECDEF折叠状态
        pageState.collapsedElements.secdefs = [];
        document.querySelectorAll('.SECDEF').forEach(secdef => {
            if (secdef.classList.contains('collapsed')) {
                const secnr = secdef.querySelector('.SECNR');
                if (secnr) {
                    // 保存SECNR和父级信息以确保唯一性
                    const sectionId = secdef.closest('.Section')?.id || '';
                    pageState.collapsedElements.secdefs.push({
                        secnr: secnr.textContent.trim(),
                        sectionId: sectionId
                    });
                }
            }
        });
        
        // 保存EXP折叠状态 - 改进版本
        pageState.collapsedElements.exponents = [];
        document.querySelectorAll('.EXP').forEach(exp => {
            if (exp.classList.contains('collapsed')) {
                // 保存更多信息以确保唯一性
                const section = exp.closest('.Section');
                const secdef = exp.closest('.Section')?.querySelector('.SECDEF');
                const secnr = secdef?.querySelector('.SECNR');
                
                pageState.collapsedElements.exponents.push({
                    text: exp.textContent.trim(),
                    sectionId: section?.id || '',
                    secnr: secnr?.textContent.trim() || ''
                });
            }
        });
        
        // 存储到sessionStorage（使用页面特定的key）
        sessionStorage.setItem(currentPageKey, JSON.stringify(pageState));
    }

    // 从sessionStorage恢复页面状态
    function restorePageState() {
        const savedState = sessionStorage.getItem(currentPageKey);
        
        if (!savedState) {
            // 如果没有保存的状态，应用默认折叠
            applyDefaultCollapse();
            return;
        }
        
        const state = JSON.parse(savedState);
        
        // 恢复滚动位置
        if (state.scrollPosition) {
            window.scrollTo(0, state.scrollPosition);
        }
        
        // 先展开所有元素，然后根据保存状态折叠特定元素
        expandAllTemporarily(() => {
            // 恢复References折叠状态
            if (state.collapsedElements.references.length > 0) {
                document.querySelectorAll('.References .TITLE').forEach(title => {
                    const titleText = title.textContent.trim();
                    if (state.collapsedElements.references.includes(titleText)) {
                        // 直接修改类，不触发点击事件
                        title.classList.add('collapsed');
                        const referencesContainer = title.closest('.References');
                        const references = Array.from(referencesContainer.querySelectorAll('.Reference, .Crossref'));
                        let nextElement = referencesContainer.nextElementSibling;
                        while (nextElement && (nextElement.classList.contains('Reference') || nextElement.classList.contains('Crossref'))) {
                            references.push(nextElement);
                            nextElement = nextElement.nextElementSibling;
                        }
                        const referencesToHide = references.filter(ref => ref !== title);
                        referencesToHide.forEach(ref => {
                            ref.classList.add('collapsed');
                        });
                    }
                });
            }
            
            // 恢复SECDEF折叠状态
            if (state.collapsedElements.secdefs.length > 0) {
                document.querySelectorAll('.SECDEF').forEach(secdef => {
                    const secnr = secdef.querySelector('.SECNR');
                    if (secnr) {
                        const sectionId = secdef.closest('.Section')?.id || '';
                        const secnrText = secnr.textContent.trim();
                        
                        // 查找匹配的保存状态
                        const shouldCollapse = state.collapsedElements.secdefs.some(item => 
                            item.secnr === secnrText && 
                            (item.sectionId === '' || item.sectionId === sectionId)
                        );
                        
                        if (shouldCollapse) {
                            // 直接修改类，不触发点击事件
                            secdef.classList.add('collapsed');
                            const currentSection = secdef.closest('.Section');
                            const exponents = currentSection.querySelectorAll('.Exponent');
                            exponents.forEach(exp => {
                                exp.classList.add('collapsed');
                            });
                        }
                    }
                });
            }
            
            // 恢复EXP折叠状态 - 改进版本
            if (state.collapsedElements.exponents.length > 0) {
                // 等待一下确保所有EXP元素都已加载
                setTimeout(() => {
                    document.querySelectorAll('.EXP').forEach(exp => {
                        const expText = exp.textContent.trim();
                        const section = exp.closest('.Section');
                        const secdef = section?.querySelector('.SECDEF');
                        const secnr = secdef?.querySelector('.SECNR');
                        
                        const sectionId = section?.id || '';
                        const secnrText = secnr?.textContent.trim() || '';
                        
                        // 查找匹配的保存状态
                        const shouldCollapse = state.collapsedElements.exponents.some(item => 
                            item.text === expText && 
                            (item.sectionId === '' || item.sectionId === sectionId) &&
                            (item.secnr === '' || item.secnr === secnrText)
                        );
                        
                        if (shouldCollapse) {
                            // 直接修改类，不触发点击事件
                            exp.classList.add('collapsed');
                        }
                    });
                }, 100);
            }
        });
    }

    // 在页面跳转前保存状态（使用更可靠的方式）
    document.addEventListener('click', function(e) {
        const link = e.target.closest('a');
        if (link && link.href && link.href !== window.location.href && !link.href.includes('#')) {
            // 立即保存状态
            savePageState();
        }
    });

    // 页面加载时尝试恢复状态
    window.addEventListener('load', function() {
        // 等待所有资源加载完成
        if (document.readyState === 'complete') {
            // 给更多时间确保所有嵌套内容加载完成
            setTimeout(restorePageState, 300);
        }
    });

    // 使用pageshow事件处理浏览器前进/后退
    window.addEventListener('pageshow', function(event) {
        if (event.persisted) {
            // 页面是从bfcache中恢复的
            setTimeout(restorePageState, 200);
        }
    });

    // 添加全局展开/折叠按钮功能
    const expandAllBtn = document.getElementById('expandAllBtn');
    const collapseAllBtn = document.getElementById('collapseAllBtn');
    
    if (expandAllBtn && collapseAllBtn) {
        expandAllBtn.addEventListener('click', function() {
            // 展开所有References
            document.querySelectorAll('.References .TITLE').forEach(title => {
                title.classList.remove('collapsed');
                const referencesContainer = title.closest('.References');
                const references = Array.from(referencesContainer.querySelectorAll('.Reference, .Crossref'));
                let nextElement = referencesContainer.nextElementSibling;
                while (nextElement && (nextElement.classList.contains('Reference') || nextElement.classList.contains('Crossref'))) {
                    references.push(nextElement);
                    nextElement = nextElement.nextElementSibling;
                }
                const referencesToShow = references.filter(ref => ref !== title);
                referencesToShow.forEach(ref => {
                    ref.classList.remove('collapsed');
                });
            });
            
            // 展开所有SECDEF和Exponent
            document.querySelectorAll('.SECDEF').forEach(header => {
                header.classList.remove('collapsed');
                const currentSection = header.closest('.Section');
                const exponents = currentSection.querySelectorAll('.Exponent');
                exponents.forEach(exp => {
                    exp.classList.remove('collapsed');
                });
            });
            
            // 展开所有EXP元素的内容
            document.querySelectorAll('.EXP').forEach(exp => {
                exp.classList.remove('collapsed');
            });
            
            // 保存状态
            savePageState();
        });
        
        collapseAllBtn.addEventListener('click', function() {
            // 折叠所有References
            document.querySelectorAll('.References .TITLE').forEach(title => {
                title.classList.add('collapsed');
                const referencesContainer = title.closest('.References');
                const references = Array.from(referencesContainer.querySelectorAll('.Reference, .Crossref'));
                let nextElement = referencesContainer.nextElementSibling;
                while (nextElement && (nextElement.classList.contains('Reference') || nextElement.classList.contains('Crossref'))) {
                    references.push(nextElement);
                    nextElement = nextElement.nextElementSibling;
                }
                const referencesToHide = references.filter(ref => ref !== title);
                referencesToHide.forEach(ref => {
                    ref.classList.add('collapsed');
                });
            });
            
            // 折叠所有SECDEF和Exponent
            document.querySelectorAll('.SECDEF').forEach(header => {
                header.classList.add('collapsed');
                const currentSection = header.closest('.Section');
                const exponents = currentSection.querySelectorAll('.Exponent');
                exponents.forEach(exp => {
                    exp.classList.add('collapsed');
                });
            });
            
            // 折叠所有EXP元素的内容
            document.querySelectorAll('.EXP').forEach(exp => {
                exp.classList.add('collapsed');
            });
            
            // 保存状态
            savePageState();
        });
    }
    
    // 监听DOM变化，确保新加载的内容也会被处理
    const observer = new MutationObserver(function(mutations) {
        mutations.forEach(function(mutation) {
            if (mutation.addedNodes.length) {
                const hasGramElements = Array.from(mutation.addedNodes).some(node => 
                    node.nodeType === 1 && (node.classList.contains('GRAM') || node.querySelector('.GRAM'))
                );
                const hasExpElements = Array.from(mutation.addedNodes).some(node => 
                    node.nodeType === 1 && (node.classList.contains('EXP') || node.querySelector('.EXP'))
                );
                
                if (hasGramElements) {
                    setTimeout(replaceGramContent, 100);
                }
                if (hasExpElements) {
                    setTimeout(handleMultiWordExp, 100);
                    // 如果有新的EXP元素加载，重新检查状态恢复
                    setTimeout(() => {
                        const savedState = sessionStorage.getItem(currentPageKey);
                        if (savedState) {
                            const state = JSON.parse(savedState);
                            if (state.collapsedElements.exponents.length > 0) {
                                restoreEXPState(state);
                            }
                        }
                    }, 150);
                }
            }
        });
    });
    
    observer.observe(document.body, {
        childList: true,
        subtree: true
    });

    // 专门的EXP状态恢复函数
    function restoreEXPState(state) {
        document.querySelectorAll('.EXP').forEach(exp => {
            const expText = exp.textContent.trim();
            const section = exp.closest('.Section');
            const secdef = section?.querySelector('.SECDEF');
            const secnr = secdef?.querySelector('.SECNR');
            
            const sectionId = section?.id || '';
            const secnrText = secnr?.textContent.trim() || '';
            
            // 查找匹配的保存状态
            const shouldCollapse = state.collapsedElements.exponents.some(item => 
                item.text === expText && 
                (item.sectionId === '' || item.sectionId === sectionId) &&
                (item.secnr === '' || item.secnr === secnrText)
            );
            
            if (shouldCollapse && !exp.classList.contains('collapsed')) {
                exp.classList.add('collapsed');
            }
        });
    }

    // 初始恢复状态 - 给更多时间确保所有内容加载
    setTimeout(() => {
        const savedState = sessionStorage.getItem(currentPageKey);
        if (savedState) {
            const state = JSON.parse(savedState);
            if (state.collapsedElements.exponents.length > 0) {
                // 额外恢复EXP状态
                setTimeout(() => restoreEXPState(state), 200);
            }
        }
    }, 800);

});